<?php
// template/ahkweb/check_status.php

session_start();
header('Content-Type: application/json');

// Disable error display to prevent HTML in JSON response
error_reporting(0);
ini_set('display_errors', 0);

require_once '../../config.php';

// Robust path loading for gateway_config.php
$gateway_config_paths = [
    __DIR__ . '/../../gateway_config.php',    // Relative to template/ahkweb
    $_SERVER['DOCUMENT_ROOT'] . '/gateway_config.php'
];

$gateway_config_loaded = false;
foreach ($gateway_config_paths as $path) {
    if (file_exists($path)) {
        require_once $path;
        $gateway_config_loaded = true;
        break;
    }
}

// Default values if gateway config not found
$selectedGatewayKey = 'pays';
$apiToken     = "2667c6bd4345bf97fc9ecfb934cdb21f";
$merchantMID  = "LYMaUK11988501129167";
$gatewayLabel = 'Gateway 1 - FATIMA';

// Validate transaction ID
if (empty($_GET['txn_id'])) {
    echo json_encode(['status'=>'error','message'=>'Missing txn_id parameter.']);
    exit;
}
$txn_id = $_GET['txn_id'];

// Use global connection if available, otherwise assume config.php set it up
global $conn; 
// config.php usually sets $conn (mysqli) or similar. 
// ahkweb usually uses $conn or $ahk_conn. 
// Let's check config.php content later if needed, but assuming $conn is standard.
// If config.php uses $ahk_conn, we might need to alias it.
if (!isset($conn) && isset($ahk_conn)) {
    $conn = $ahk_conn;
}

if (!$conn) {
    echo json_encode(['status'=>'error','message'=>'DB connection failed.']);
    exit;
}

// Fetch transaction
// Using mysqli as ahkweb/wallet.php uses mysqli
$stmt = $conn->prepare("SELECT * FROM wallet WHERE txn_id = ? LIMIT 1");
$stmt->bind_param("s", $txn_id);
$stmt->execute();
$result = $stmt->get_result();
$transaction = $result->fetch_assoc();

if (!$transaction) {
    echo json_encode(['status'=>'error','message'=>'Transaction not found.']);
    exit;
}

// DETERMINE GATEWAY
$selectedGatewayKey = 'pays'; // Default fallback
$gatewayConfig = [];

// 0. Fetch DB Settings first
$db_settings_q = $conn->query("SELECT upi_id, upi_merchant_code, upi_secret FROM settings WHERE id = 1");
$db_settings = ($db_settings_q && $db_settings_q->num_rows > 0) ? $db_settings_q->fetch_assoc() : null;

// 1. Prefer passed gateway parameter
if (!empty($_GET['gateway'])) {
    $requestedGateway = $_GET['gateway'];
    if ($requestedGateway === 'db_settings' && $db_settings) {
        $selectedGatewayKey = 'db_settings';
        $gatewayConfig = [
            'merchant_mid' => $db_settings['upi_merchant_code'],
            'api_token'    => $db_settings['upi_secret'],
            'merchant_upi' => $db_settings['upi_id']
        ];
    } elseif (function_exists('ld_get_gateways')) {
        $allGateways = ld_get_gateways();
        if (isset($allGateways[$requestedGateway])) {
            $selectedGatewayKey = $requestedGateway;
            $gatewayConfig = $allGateways[$selectedGatewayKey];
        }
    }
} 
// 2. Fallback to transaction lookup
elseif (function_exists('ld_lookup_gateway_for_txn')) {
    $lookupKey = ld_lookup_gateway_for_txn($txn_id);
    if ($lookupKey === 'db_settings' && $db_settings) {
        $selectedGatewayKey = 'db_settings';
        $gatewayConfig = [
            'merchant_mid' => $db_settings['upi_merchant_code'],
            'api_token'    => $db_settings['upi_secret'],
            'merchant_upi' => $db_settings['upi_id']
        ];
    } elseif ($lookupKey && function_exists('ld_get_gateways')) {
        $allGateways = ld_get_gateways();
        if (isset($allGateways[$lookupKey])) {
            $selectedGatewayKey = $lookupKey;
            $gatewayConfig = $allGateways[$selectedGatewayKey];
        }
    }
}

// 3. Fallback to active gateway or DB default
if (empty($gatewayConfig)) {
    // If DB settings exist, use them as default preference
    if ($db_settings && !empty($db_settings['upi_merchant_code']) && !empty($db_settings['upi_secret'])) {
        $selectedGatewayKey = 'db_settings';
        $gatewayConfig = [
            'merchant_mid' => $db_settings['upi_merchant_code'],
            'api_token'    => $db_settings['upi_secret'],
            'merchant_upi' => $db_settings['upi_id']
        ];
    } elseif (function_exists('ld_get_active_gateway')) {
        $gatewayConfig = ld_get_active_gateway();
        $selectedGatewayKey = $gatewayConfig['key'] ?? 'pays';
    }
}

$merchantMID = $gatewayConfig['merchant_mid'] ?? $merchantMID;
$apiToken = $gatewayConfig['api_token'] ?? $apiToken;

// If already processed
if ($transaction['status'] === 'success' || $transaction['status'] === 'paid') {
    echo json_encode([
        'status' => 'paid',
        'message' => 'Payment was already processed successfully.'
    ]);
    exit;
}

// Check payment status with Gateway (Paytm API)
// Using logic from new_website_api check_status.php
$paytmResponse = @file_get_contents(
    "https://securegw.paytm.in/order/status?JsonData=" . 
    urlencode(json_encode([
        'MID'     => $merchantMID,
        'ORDERID' => $txn_id
    ])),
    false,
    stream_context_create([
        'http' => [
            'method'  => 'GET',
            'timeout' => 30
        ]
    ])
);

if ($paytmResponse) {
    $responseData = json_decode($paytmResponse, true);
    
    // Paytm Success Status: "TXN_SUCCESS"
    if (isset($responseData['STATUS']) && $responseData['STATUS'] === 'TXN_SUCCESS') {
        
        // Verify Amount
        $paidAmount = floatval($responseData['TXNAMOUNT'] ?? 0);
        $txnAmount = floatval($transaction['amount']);
        
        if ($paidAmount >= $txnAmount) {
            // Update payment status
            $updateStmt = $conn->prepare("UPDATE wallet SET status = 'success' WHERE txn_id = ?");
            $updateStmt->bind_param("s", $txn_id);
            $updateStmt->execute();

            // Update user balance
            // Need to find user by phone or email as in wallet.php
            $phone = $transaction['phone'];
            
            // Get current balance
            $balStmt = $conn->prepare("SELECT balance FROM users WHERE mobile = ? LIMIT 1");
            $balStmt->bind_param("s", $phone);
            $balStmt->execute();
            $balRes = $balStmt->get_result();
            if ($balRes->num_rows > 0) {
                $userRow = $balRes->fetch_assoc();
                $oldBal = floatval($userRow['balance']);
                $newBal = $oldBal + $txnAmount;
                
                // Update balance
                $updateUser = $conn->prepare("UPDATE users SET balance = ? WHERE mobile = ?");
                $updateUser->bind_param("ds", $newBal, $phone);
                $updateUser->execute();
            }

            echo json_encode([
                'status' => 'paid',
                'message' => 'Payment verified successfully.'
            ]);
            exit;
        } else {
             echo json_encode(['status'=>'pending','message'=>'Amount mismatch. Paid: '.$paidAmount]);
             exit;
        }
    } else {
        echo json_encode([
            'status' => 'pending', 
            'message' => $responseData['RESPMSG'] ?? 'Payment pending'
        ]);
        exit;
    }
} else {
    echo json_encode(['status'=>'pending','message'=>'Could not connect to gateway.']);
    exit;
}
?>
